#!/bin/bash

# Script to process domains from a file, validate format, check if they exist, and delete them

# Check if the input file exists
if [ ! -f "$1" ]; then
    echo "Error: Input file '$1' not found."
    exit 1
fi

# Define a regex pattern for valid domain format
# This pattern checks for a basic domain format: something.something
DOMAIN_PATTERN="^[a-zA-Z0-9][a-zA-Z0-9-]*\.[a-zA-Z0-9][a-zA-Z0-9-]*(\.[a-zA-Z0-9-]+)*$"

# Read the file line by line
while IFS= read -r domain || [ -n "$domain" ]; do
    # Skip empty lines
    if [ -z "$domain" ]; then
        continue
    fi

    # Check if the domain has a valid format
    if [[ $domain =~ $DOMAIN_PATTERN ]]; then
        echo "Processing valid domain: $domain"

        # Check if the domain exists using doctl
        if doctl compute domain get "$domain" &>/dev/null; then
            echo "Domain '$domain' exists. Deleting..."

            # Delete the domain
            if doctl compute domain delete "$domain" -f; then
                echo "Domain '$domain' successfully deleted."
            else
                echo "Failed to delete domain '$domain'."
            fi
        else
            echo "Domain '$domain' does not exist in DigitalOcean."
        fi
    else
        echo "Invalid domain format: $domain (skipping)"
    fi
done < "a"

echo "Domain processing complete."